<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class CustomerController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Customer::query();
        
        if ($request->has('search')) {
            $query->where('name', 'like', "%{$request->search}%")
                  ->orWhere('document_number', 'like', "%{$request->search}%")
                  ->orWhere('email', 'like', "%{$request->search}%")
                  ->orWhere('phone', 'like', "%{$request->search}%");
        }
        
        if ($request->has('active')) {
            $query->where('is_active', $request->boolean('active'));
        }
        
        if ($request->has('document_type')) {
            $query->where('document_type', $request->document_type);
        }
        
        $customers = $query->orderBy('name')->paginate(10);
        
        return response()->json($customers);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'document_type' => 'nullable|string|max:20',
            'document_number' => 'nullable|string|max:50',
            'email' => 'nullable|email',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'notes' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $customer = Customer::create($validated);
        
        return response()->json($customer, 201);
    }

    public function show(string $id): JsonResponse
    {
        $customer = Customer::with(['sales', 'invoices'])->findOrFail($id);
        return response()->json($customer);
    }

    public function getByDocument(Request $request): JsonResponse
    {
        $documentNumber = $request->query('document');
        
        if (!$documentNumber) {
            return response()->json(null);
        }
        
        $customer = Customer::where('document_number', $documentNumber)->first();
        
        return response()->json($customer);
    }

    public function update(Request $request, string $id): JsonResponse
    {
        $customer = Customer::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'document_type' => 'nullable|string|max:20',
            'document_number' => 'nullable|string|max:50',
            'email' => 'nullable|email',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'notes' => 'nullable|string',
            'is_active' => 'boolean',
        ]);
        
        $customer->update($validated);
        
        return response()->json($customer);
    }

    public function destroy(string $id): JsonResponse
    {
        $customer = Customer::findOrFail($id);
        
        if ($customer->sales()->count() > 0 || $customer->invoices()->count() > 0) {
            return response()->json(['error' => 'Cannot delete customer with associated sales or invoices'], 400);
        }
        
        $customer->delete();
        
        return response()->json(['message' => 'Customer deleted successfully']);
    }

    public function byDocument(string $documentNumber): JsonResponse
    {
        $customer = Customer::where('document_number', $documentNumber)->first();
        
        if (!$customer) {
            return response()->json(null);
        }
        
        return response()->json($customer);
    }
}
