<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Inventory;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class InventoryController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Inventory::with(['product', 'user']);
        
        if ($request->has('product_id')) {
            $query->where('product_id', $request->product_id);
        }
        
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }
        
        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        //$inventories = $query->orderBy('created_at', 'desc')->paginate(20);
        $inventories = $query->orderBy('created_at', 'desc')->get();
        
        return response()->json($inventories);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'type' => 'required|in:in,out,adjustment',
            'quantity' => 'required|integer',
            'notes' => 'nullable|string',
        ]);

        $product = Product::findOrFail($validated['product_id']);
        $previousStock = $product->stock;

        if ($validated['type'] === 'out' && $previousStock < $validated['quantity']) {
            return response()->json(['error' => 'Insufficient stock'], 400);
        }

        if ($validated['type'] === 'in') {
            $newStock = $previousStock + $validated['quantity'];
        } elseif ($validated['type'] === 'out') {
            $newStock = $previousStock - $validated['quantity'];
        } else {
            $newStock = $validated['quantity'];
        }

        $product->update(['stock' => $newStock]);

        $inventory = Inventory::create([
            'product_id' => $validated['product_id'],
            'user_id' => Auth::id(),
            'type' => $validated['type'],
            'quantity' => $validated['quantity'],
            'previous_stock' => $previousStock,
            'new_stock' => $newStock,
            'notes' => $validated['notes'] ?? null,
        ]);

        return response()->json($inventory->load(['product', 'user']), 201);
    }

    public function show(string $id): JsonResponse
    {
        $inventory = Inventory::with(['product', 'user'])->findOrFail($id);
        return response()->json($inventory);
    }

    public function update(Request $request, string $id): JsonResponse
    {
        return response()->json(['error' => 'Inventory adjustments cannot be modified'], 400);
    }

    public function destroy(string $id): JsonResponse
    {
        return response()->json(['error' => 'Inventory records cannot be deleted'], 400);
    }
}
