<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\File;

class ProductController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Product::with(['category', 'supplier']);
        
        if ($request->has('search')) {
            $query->where('name', 'like', "%{$request->search}%")
                  ->orWhere('sku', 'like', "%{$request->search}%")
                  ->orWhere('barcode', 'like', "%{$request->search}%");
        }
        
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }
        
        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }
        
        if ($request->has('active')) {
            $query->where('is_active', $request->boolean('active'));
        }
        
        if ($request->has('low_stock')) {
            $query->whereRaw('stock <= min_stock');
        }
        
        //$products = $query->orderBy('name')->paginate(10);
        $products = $query->orderBy('name')->get();
        
        return response()->json($products);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'required|string|unique:products,sku',
            'barcode' => 'nullable|string|unique:products,barcode',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'purchase_price' => 'required|numeric|min:0',
            'sale_price' => 'required|numeric|min:0',
            'stock' => 'integer|min:0',
            'min_stock' => 'integer|min:0',
            'max_stock' => 'integer|min:0',
            'unit' => 'string|max:50',
            'image' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $product = Product::create($validated);
        
        return response()->json($product->load(['category', 'supplier']), 201);
    }

    public function show(string $id): JsonResponse
    {
        $product = Product::with(['category', 'supplier', 'inventories', 'purchaseItems', 'saleItems'])->findOrFail($id);
        return response()->json($product);
    }

    public function update(Request $request, string $id): JsonResponse
    {
        $product = Product::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'sku' => 'sometimes|string|unique:products,sku,' . $id,
            'barcode' => 'nullable|string|unique:products,barcode,' . $id,
            'description' => 'nullable|string',
            'category_id' => 'sometimes|exists:categories,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'purchase_price' => 'sometimes|numeric|min:0',
            'sale_price' => 'sometimes|numeric|min:0',
            'stock' => 'integer|min:0',
            'min_stock' => 'integer|min:0',
            'max_stock' => 'integer|min:0',
            'unit' => 'string|max:50',
            'image' => 'nullable|string',
            'is_active' => 'boolean',
        ]);
        
        $product->update($validated);
        
        return response()->json($product->load(['category', 'supplier']));
    }

    public function uploadImage(Request $request, string $id): JsonResponse
    {
        $product = Product::findOrFail($id);

        $validated = $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        if ($product->image) {
            $oldPath = storage_path('app/public/' . $product->image);
            if (File::exists($oldPath)) {
                File::delete($oldPath);
            }
        }

        $path = $request->file('image')->store('products', 'public');

        $product->update(['image' => $path]);

        return response()->json([
            'message' => 'Imagen actualizada correctamente',
            'image' => $path
        ]);
    }

    public function deleteImage(string $id): JsonResponse
    {
        $product = Product::findOrFail($id);

        if ($product->image) {
            $path = storage_path('app/public/' . $product->image);
            if (File::exists($path)) {
                File::delete($path);
            }
            $product->update(['image' => null]);
        }

        return response()->json(['message' => 'Imagen eliminada correctamente']);
    }

    public function destroy(string $id): JsonResponse
    {
        $product = Product::findOrFail($id);
        
        if ($product->image) {
            $path = storage_path('app/public/' . $product->image);
            if (File::exists($path)) {
                File::delete($path);
            }
        }
        
        $product->delete();
        
        return response()->json(['message' => 'Product deleted successfully']);
    }
}
