<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Inventory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class PurchaseController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Purchase::with(['supplier', 'user', 'items.product']);
        
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }
        
        if ($request->has('date_from')) {
            $query->where('purchase_date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->where('purchase_date', '<=', $request->date_to);
        }
        
        //$purchases = $query->orderBy('purchase_date', 'desc')->paginate(10);
        $purchases = $query->orderBy('purchase_date', 'desc')->get();
        
        return response()->json($purchases);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'purchase_date' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'tax' => 'numeric|min:0',
            'discount' => 'numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        $validated['user_id'] = Auth::id();
        $validated['purchase_number'] = 'PUR-' . date('YmdHis');
        
        $subtotal = 0;
        foreach ($validated['items'] as &$item) {
            $item['subtotal'] = $item['quantity'] * $item['unit_price'];
            $subtotal += $item['subtotal'];
        }
        unset($item);
        
        $validated['subtotal'] = $subtotal;
        $validated['tax'] = $validated['tax'] ?? 0;
        $validated['discount'] = $validated['discount'] ?? 0;
        $validated['total'] = $subtotal + $validated['tax'] - $validated['discount'];
        $validated['status'] = 'pending';

        $purchase = Purchase::create($validated);

        foreach ($validated['items'] as $item) {
            PurchaseItem::create([
                'purchase_id' => $purchase->id,
                'product_id' => $item['product_id'],
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'subtotal' => $item['subtotal'],
            ]);
        }

        return response()->json($purchase->load(['supplier', 'user', 'items.product']), 201);
    }

    public function show(string $id): JsonResponse
    {
        $purchase = Purchase::with(['supplier', 'user', 'items.product'])->findOrFail($id);
        return response()->json($purchase);
    }

    public function update(Request $request, string $id): JsonResponse
    {
        $purchase = Purchase::findOrFail($id);
        
        if ($purchase->status === 'completed') {
            return response()->json(['error' => 'Cannot update a completed purchase'], 400);
        }

        $validated = $request->validate([
            'status' => 'sometimes|in:pending,completed,cancelled',
            'notes' => 'nullable|string',
        ]);
        
        $purchase->update($validated);
        
        return response()->json($purchase->load(['supplier', 'user', 'items.product']));
    }

    public function destroy(string $id): JsonResponse
    {
        $purchase = Purchase::findOrFail($id);
        
        if ($purchase->status === 'completed') {
            return response()->json(['error' => 'Cannot delete a completed purchase'], 400);
        }

        foreach ($purchase->items as $item) {
            $product = $item->product;
            $product->update(['stock' => $product->stock - $item->quantity]);
            $item->delete();
        }
        
        $purchase->delete();
        
        return response()->json(['message' => 'Purchase deleted successfully']);
    }

    public function complete(string $id): JsonResponse
    {
        $purchase = Purchase::findOrFail($id);
        
        if ($purchase->status !== 'pending') {
            return response()->json(['error' => 'Only pending purchases can be completed'], 400);
        }

        foreach ($purchase->items as $item) {
            $product = $item->product;
            $previousStock = $product->stock;
            $newStock = $previousStock + $item->quantity;
            
            $product->update(['stock' => $newStock]);

            Inventory::create([
                'product_id' => $product->id,
                'user_id' => Auth::id(),
                'type' => 'in',
                'quantity' => $item->quantity,
                'previous_stock' => $previousStock,
                'new_stock' => $newStock,
                'reference_type' => 'purchase',
                'reference_id' => $purchase->id,
                'notes' => 'Purchase completed: ' . $purchase->purchase_number,
            ]);
        }
        
        $purchase->update(['status' => 'completed']);
        
        return response()->json($purchase->load(['supplier', 'user', 'items.product']));
    }

    public function cancel(string $id): JsonResponse
    {
        $purchase = Purchase::findOrFail($id);
        
        if ($purchase->status !== 'pending') {
            return response()->json(['error' => 'Only pending purchases can be cancelled'], 400);
        }
        
        $purchase->update(['status' => 'cancelled']);
        
        return response()->json($purchase->load(['supplier', 'user', 'items.product']));
    }
}
