<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Settings;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class SettingsController extends Controller
{
    public function index(): JsonResponse
    {
        $settings = Settings::getSettings();
        return response()->json($settings);
    }

    public function update(Request $request): JsonResponse
    {
        $settings = Settings::getSettings();

        $validated = $request->validate([
            'company_name' => 'nullable|string|max:255',
            'company_address' => 'nullable|string',
            'company_phone' => 'nullable|string|max:50',
            'company_contact' => 'nullable|string|max:255',
            'company_logo' => 'nullable|string|max:500',
            'currency_local_name' => 'nullable|string|max:50',
            'currency_local_symbol' => 'nullable|string|max:10',
            'currency_usd_name' => 'nullable|string|max:50',
            'currency_usd_symbol' => 'nullable|string|max:10',
            'exchange_rate' => 'nullable|numeric|min:0',
            'idle_time_minutes' => 'nullable|integer|min:1|max:60',
            'auto_backup_enabled' => 'nullable|boolean',
            'auto_backup_time' => 'nullable',
            'backup_path' => 'nullable|string|max:500',
        ]);

        $settings->update($validated);

        return response()->json($settings);
    }

    public function createBackup(Request $request): JsonResponse
    {
        try {
            $filename = 'backup_' . Carbon::now()->format('Y-m-d_His') . '.sql';
            $path = storage_path('app/backups/' . $filename);

            if (!File::exists(storage_path('app/backups'))) {
                File::makeDirectory(storage_path('app/backups'), 0755, true);
            }

            $database = config('database.connections.mysql.database');
            $username = config('database.connections.mysql.username');
            $password = config('database.connections.mysql.password');
            $host = config('database.connections.mysql.host');

            $command = sprintf(
                'mysqldump -u%s %s %s > "%s"',
                $username,
                $password ? '-p' . escapeshellarg($password) : '',
                $database,
                $path
            );

            if ($password) {
                $command = sprintf(
                    'mysqldump -u%s -p%s %s > "%s"',
                    $username,
                    $password,
                    $database,
                    $path
                );
            } else {
                $command = sprintf(
                    'mysqldump -u%s %s > "%s"',
                    $username,
                    $database,
                    $path
                );
            }

            exec($command, $output, $result);

            if ($result !== 0 || !File::exists($path)) {
                return response()->json(['error' => 'Error al crear el respaldo'], 500);
            }

            return response()->json([
                'message' => 'Respaldo creado correctamente',
                'filename' => $filename,
                'path' => 'backups/' . $filename,
                'created_at' => Carbon::now()->format('Y-m-d H:i:s'),
                'size' => File::size($path)
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al crear el respaldo: ' . $e->getMessage()], 500);
        }
    }

    public function listBackups(): JsonResponse
    {
        $backupsPath = storage_path('app/backups');
        
        if (!File::exists($backupsPath)) {
            return response()->json([]);
        }

        $files = File::files($backupsPath);
        $backups = [];

        foreach ($files as $file) {
            if (pathinfo($file->getFilename(), PATHINFO_EXTENSION) === 'sql') {
                $backups[] = [
                    'filename' => $file->getFilename(),
                    'path' => 'backups/' . $file->getFilename(),
                    'created_at' => Carbon::createFromTimestamp($file->getMTime())->format('Y-m-d H:i:s'),
                    'size' => $file->getSize(),
                    'size_formatted' => $this->formatBytes($file->getSize())
                ];
            }
        }

        usort($backups, function ($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });

        return response()->json($backups);
    }

    public function downloadBackup(Request $request): BinaryFileResponse|JsonResponse
    {
        $filename = $request->input('filename');
        $path = storage_path('app/backups/' . $filename);

        if (!File::exists($path)) {
            return response()->json(['error' => 'Respaldo no encontrado'], 404);
        }

        return response()->download($path, $filename);
    }

    public function deleteBackup(Request $request): JsonResponse
    {
        $filename = $request->input('filename');
        $path = storage_path('app/backups/' . $filename);

        if (!File::exists($path)) {
            return response()->json(['error' => 'Respaldo no encontrado'], 404);
        }

        File::delete($path);

        return response()->json(['message' => 'Respaldo eliminado correctamente']);
    }

    private function formatBytes($bytes, $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, $precision) . ' ' . $units[$pow];
    }
}
